<?php
// Simple PO -> MO compiler
// Usage: php generate_mo.php path/to/file.po [path/to/file.mo]

if ($argc < 2) {
    echo "Usage: php generate_mo.php file.po [file.mo]\n";
    exit(1);
}

$poFile = $argv[1];
$moFile = $argv[2] ?? preg_replace('/\.po$/', '.mo', $poFile);

if (!file_exists($poFile)) {
    echo "PO file not found: $poFile\n";
    exit(2);
}

$contents = file_get_contents($poFile);
$lines = preg_split('/\r?\n/', $contents);

$entries = [];
$state = null;
$cur = ['msgid' => '', 'msgstr' => ''];

foreach ($lines as $line) {
    $line = trim($line);
    if ($line === '' || strpos($line, '#') === 0) {
        if ($cur['msgid'] !== '' || $cur['msgstr'] !== '') {
            $entries[] = $cur;
            $cur = ['msgid' => '', 'msgstr' => ''];
            $state = null;
        }
        continue;
    }

    if (preg_match('/^msgid\s+"(.*)"$/', $line, $m)) {
        $state = 'msgid';
        $cur['msgid'] = stripcslashes($m[1]);
        continue;
    }
    if (preg_match('/^msgstr\s+"(.*)"$/', $line, $m)) {
        $state = 'msgstr';
        $cur['msgstr'] = stripcslashes($m[1]);
        continue;
    }
    if (preg_match('/^"(.*)"$/', $line, $m)) {
        if ($state && isset($cur[$state])) {
            $cur[$state] .= stripcslashes($m[1]);
        }
        continue;
    }
}
// push last
if ($cur['msgid'] !== '' || $cur['msgstr'] !== '') {
    $entries[] = $cur;
}

// Build translations map, skipping empty msgid header
$trans = [];
foreach ($entries as $e) {
    $id = $e['msgid'];
    $str = $e['msgstr'];
    if ($id === '') { // header
        continue;
    }
    $trans[$id] = $str;
}

ksort($trans);

// Build MO binary
$ids = array_keys($trans);
$strings = array_values($trans);

$offsets = [];
$idsBuf = '';
$stringsBuf = '';

foreach ($ids as $id) {
    $idsBuf .= $id . "\0";
}
foreach ($strings as $s) {
    $stringsBuf .= $s . "\0";
}

$idsLen = strlen($idsBuf);
$stringsLen = strlen($stringsBuf);

$originals = [];
$translations = [];

$curIdOffset = 0;
foreach ($ids as $id) {
    $len = strlen($id);
    $originals[] = [$len, $curIdOffset];
    $curIdOffset += $len + 1;
}

$curStrOffset = 0;
foreach ($strings as $s) {
    $len = strlen($s);
    $translations[] = [$len, $curStrOffset];
    $curStrOffset += $len + 1;
}

$of = fopen($moFile, 'wb');
if (!$of) {
    echo "Cannot open for writing: $moFile\n";
    exit(3);
}

// MO Header
$magic = 0x950412de;
$f = function($v){ return pack('V', $v); };

// header parts
$revision = 0;
$n = count($originals);
$origTableOffset = 28; // header size
$transTableOffset = $origTableOffset + ($n * 8);
$idsOffset = $transTableOffset + ($n * 8);
$stringsOffset = $idsOffset + $idsLen;

fwrite($of, $f($magic));
fwrite($of, $f($revision));
fwrite($of, $f($n));
fwrite($of, $f($origTableOffset));
fwrite($of, $f($transTableOffset));
// size of hashing table (unused)
fwrite($of, $f(0));

// original table
$cur = $idsOffset;
for ($i = 0; $i < $n; $i++) {
    fwrite($of, $f($originals[$i][0]));
    fwrite($of, $f($cur));
    $cur += $originals[$i][0] + 1;
}

// translation table
$cur = $stringsOffset;
for ($i = 0; $i < $n; $i++) {
    fwrite($of, $f($translations[$i][0]));
    fwrite($of, $f($cur));
    $cur += $translations[$i][0] + 1;
}

// write ids and strings
fwrite($of, $idsBuf);
fwrite($of, $stringsBuf);

fclose($of);

echo "Compiled MO to: $moFile\n";
exit(0);

